import axios, { AxiosInstance, AxiosResponse } from 'axios';
import {
  StartCheckoutSessionRequestDto,
  StartCheckoutSessionResponseDto,
} from './dto/start-checkout-session.dto';
import { BaseResponseDto } from './dto/common.dto';
import { GetCheckoutDetailResponseDto } from './dto/get-checkout-detail.dto';

// eslint-disable-next-line @typescript-eslint/no-var-requires
const crypto = require('crypto');

interface MISPaySDKOptions {
  env?: 'prod' | 'sandbox';
  debug?: boolean;
  baseUrl?: {
    prod: string;
    sandbox: string;
  };
}

const PROD_ENV = process.env?.MISPAY_PROD_ENV || 'https://api.mispay.co/v1/api';
const SANDBOX_ENV =
  process.env?.MISPAY_SANDBOX_ENV || 'https://api.mispay.co/sandbox/v1/api';

const defaultOptions: MISPaySDKOptions = {
  env: 'prod',
  debug: false,
  baseUrl: {
    prod: PROD_ENV,
    sandbox: SANDBOX_ENV,
  },
};

export class MISPaySDK {
  private appId: string;
  private appSecret: string;
  private http: AxiosInstance | undefined;
  private options: MISPaySDKOptions;

  private readonly endpoints = {
    token: '/token',
    startCheckoutSession: '/start-checkout',
    getCheckoutDetail: '/checkout/:id',
    endCheckoutSession: '/checkout/:id/end',
  };

  private token?: string;

  constructor(appId: string, appSecret: string, options: MISPaySDKOptions) {
    this.appId = appId;
    this.appSecret = appSecret;
    this.options = { ...defaultOptions, ...options };
    this.httpBuild();
  }

  private log(payload: any) {
    if (this.options.debug) {
      console.log(new Date(), payload);
    }
  }

  private httpBuild() {
    if (!this.appId || !this.appSecret) {
      throw new Error('Bad Parameters!');
    }
    this.http = axios.create({
      baseURL:
        this.options.env === 'prod'
          ? this.options.baseUrl.prod
          : this.options.baseUrl.sandbox,
      headers: {
        'x-app-id': this.appId,
        'x-app-secret': this.appSecret,
      },
    });
  }

  private async getToken() {
    if (this.token) {
      // TODO: check token whether is valid
      return;
    }
    try {
      if (!this.http) {
        throw new Error('not found permission!');
      }

      const response = await this.http.get(this.endpoints.token);

      this.log('register :: response: ' + JSON.stringify(response.data));

      const decryptedCode = this.decryptAES(
        response.data.result.token,
      ) as unknown as { token: string };

      this.token = decryptedCode.token;

      this.log('register :: token: ' + this.token);

      this.http.defaults.headers.common[
        'Authorization'
      ] = `Bearer ${this.token}`;
    } catch (error) {
      console.error('error' + error);
    }
  }

  private decryptAES(ciphertext: any): string {
    try {
      const input = new Buffer(ciphertext, 'base64');
      const salt = input.slice(0, 16);
      const nonce = input.slice(16, 28);
      ciphertext = input.slice(28, -16);
      const tag = input.slice(-16);
      const key = crypto.pbkdf2Sync(this.appSecret, salt, 40000, 32, 'sha256');

      const cipher = crypto.createDecipheriv('aes-256-gcm', key, nonce);
      cipher.setAuthTag(tag);
      const plaintext = Buffer.concat([
        cipher.update(ciphertext),
        cipher.final(),
      ]);

      return JSON.parse(plaintext.toString('utf-8'));
    } catch (error) {
      throw new Error('can not be decrypted!');
    }
  }

  async startCheckoutSession(
    orderId: string,
    price: number,
  ): Promise<BaseResponseDto<StartCheckoutSessionResponseDto>> {
    this.log('startCheckoutSession :: token: ' + this.token);

    await this.getToken();

    if (!this.http) {
      throw new Error('not found permission!');
    }

    const requestPayload: StartCheckoutSessionRequestDto = {
      orderId,
      purchaseAmount: price,
      purchaseCurrency: 'SAR',
    };

    const response = (await this.http.post(
      this.endpoints.startCheckoutSession,
      requestPayload,
    )) as AxiosResponse<BaseResponseDto<StartCheckoutSessionResponseDto>>;

    this.log(
      'startCheckoutSession :: response.data: ' +
        JSON.stringify(response.data.result),
    );

    return response.data;
  }

  async getCheckoutDetail(
    checkoutId: string,
  ): Promise<BaseResponseDto<GetCheckoutDetailResponseDto>> {
    this.log('getCheckoutDetail :: checkoutId: ' + checkoutId);

    await this.getToken();

    if (!this.http) {
      throw new Error('not found permission!');
    }

    const response = (await this.http.get(
      this.endpoints.getCheckoutDetail.replace(':id', checkoutId),
    )) as AxiosResponse<BaseResponseDto<GetCheckoutDetailResponseDto>>;

    this.log('getCheckoutDetail :: response: ' + JSON.stringify(response.data));
    return response.data;
  }

  async endCheckoutSession(checkoutId: string) {
    await this.getToken();

    if (!this.http) {
      throw new Error('not found permission!');
    }

    const response = await this.http.put(
      this.endpoints.endCheckoutSession.replace(':id', checkoutId),
    );

    this.log(
      'endCheckoutSession :: response: ' + JSON.stringify(response.data),
    );
  }
}
